package widget

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	tea "github.com/charmbracelet/bubbletea"
)

type DurationPromptMsg struct {
	Duration time.Duration
	Error    error
}

type DurationPrompt struct {
	Prompt string
	Input  string

	Back tea.Model
}

func NewDurationPrompt(p string, v time.Duration) DurationPrompt {
	fmt := ""
	if v != time.Duration(0) {
		fmt = v.String()
	}

	return DurationPrompt{
		Prompt: p,
		Input:  fmt,
	}
}

func (m DurationPrompt) Value() (time.Duration, error) {
	return parseDuration(m.Input)
}

func (m DurationPrompt) Init() tea.Cmd {
	return nil
}

func (m DurationPrompt) Update(msg tea.Msg) (tea.Model, tea.Cmd) {
	switch msg := msg.(type) {

	case tea.KeyMsg:
		switch msg.String() {

		case "esc":
			if m.Back != nil {
				return m.Back, func() tea.Msg {
					return DurationPromptMsg{time.Duration(0), Cancelled}
				}
			} else {
				return m, tea.Quit
			}

		case "enter":
			if m.Back != nil {
				return m.Back, func() tea.Msg {
					v, e := m.Value()
					return DurationPromptMsg{v, e}
				}
			} else {
				return m, tea.Quit
			}

		case "backspace":
			if m.Input != "" {
				m.Input = m.Input[:len(m.Input)-1]
			}

		case "delete":
			m.Input = ""

		case "alt+enter":
			m.Input += "\n"

		default:
			// milli/microseconds are not needed.
			if strings.Contains(" 0123456789hms-.", msg.String()) {
				m.Input += msg.String()
			}
		}
	}

	return m, nil
}

func (m DurationPrompt) View() string {
	p := m.Prompt

	if p == "" {
		p = "Enter duration"
	}

	s := fmt.Sprintf("\n%s:\n> %s\n\n", p, m.Input)

	if _, err := m.Value(); err != nil {
		s += err.Error() + "\n\n"
	}

	s += "Press 'alt+enter' for new line.\n"
	s += "Press 'esc' to abort.\n"
	s += "Press 'enter' to confirm.\n"

	return s
}

func parseDuration(v string) (time.Duration, error) {
	// Spaces are allowed in prompt, but not in time.parseDuration.
	v = strings.ReplaceAll(v, " ", "")

	if v == "" {
		return time.Duration(0), nil
	}

	// No unit. Assume minutes.
	if _, err := strconv.ParseInt(v, 0, 64); err == nil {
		v += "m"
	}

	return time.ParseDuration(v)
}
