package scene

import (
	"fmt"
	"net/url"
	"strings"

	"github.com/charmbracelet/bubbles/list"
	"github.com/charmbracelet/bubbles/textinput"
	tea "github.com/charmbracelet/bubbletea"
	"vnbr.de/track/internal/config"
)

type SetupWizard struct {
	Config config.Config
	back   tea.Model
	saved  bool

	ticketRepo list.Model

	jiraDomain textinput.Model
	jiraUser   textinput.Model
	jiraToken  textinput.Model
}

func NewSetupWizard(config config.Config, back tea.Model) SetupWizard {
	wizard := SetupWizard{Config: config, back: back}

	wizard.ticketRepo = list.New([]list.Item{}, list.NewDefaultDelegate(), 80, 5)
	wizard.ticketRepo.DisableQuitKeybindings()
	wizard.ticketRepo.FilterInput.Focus()
	wizard.ticketRepo.SetFilteringEnabled(false)
	wizard.ticketRepo.SetItems([]list.Item{
		ticketRepo{"file", "File", "Tickets are stored in an .ini file."},
		ticketRepo{"jira", "Jira", "Connect with Jira to fetch issues."},
	})
	wizard.ticketRepo.SetShowHelp(false)
	wizard.ticketRepo.SetShowPagination(false)
	wizard.ticketRepo.SetShowStatusBar(false)
	wizard.ticketRepo.SetShowTitle(false)
	wizard.ticketRepo.Title = "Tickets Storage:"
	wizard.ticketRepo.Select(wizard.ticketIndex(config.GetTicketManagerValue()))

	jira := config.GetJiraCredentials()
	jiraToken, _ := jira.User.Password()

	wizard.jiraDomain = textinput.New()
	wizard.jiraDomain.Prompt = "Domain: "
	wizard.jiraDomain.SetValue(jira.Host)

	wizard.jiraUser = textinput.New()
	wizard.jiraUser.Prompt = "Email: "
	wizard.jiraUser.SetValue(jira.User.Username())

	wizard.jiraToken = textinput.New()
	wizard.jiraToken.EchoMode = textinput.EchoPassword
	wizard.jiraToken.Prompt = "API Token: "
	wizard.jiraToken.SetValue(jiraToken)

	return wizard
}

func (m SetupWizard) Init() tea.Cmd {
	var cmds tea.BatchMsg

	// Dispatch Update messages, that couldn't be dispatched via New().
	cmds = append(cmds, m.ticketRepo.SetItems(m.ticketRepo.Items()))

	return tea.Batch(cmds...)
}

func (m SetupWizard) Update(msg tea.Msg) (tea.Model, tea.Cmd) {
	var cmds []tea.Cmd

	switch msg := msg.(type) {

	case tea.KeyMsg:
		if m.saved {
			return m.exit()
		}

		switch msg.String() {

		case "esc":
			return m.focusPrev()

		case "enter":
			return m.focusNext()

		case "ctrl+s":
			m.save()
			m.saved = true
			return m, nil

		}
	}

	var cmd tea.Cmd
	if m.ticketRepo.FilterInput.Focused() {
		m.ticketRepo, cmd = m.ticketRepo.Update(msg)
		cmds = append(cmds, cmd)
	} else if m.jiraDomain.Focused() {
		m.jiraDomain, cmd = m.jiraDomain.Update(msg)
		cmds = append(cmds, cmd)
	} else if m.jiraUser.Focused() {
		m.jiraUser, cmd = m.jiraUser.Update(msg)
		cmds = append(cmds, cmd)
	} else if m.jiraToken.Focused() {
		m.jiraToken, cmd = m.jiraToken.Update(msg)
		cmds = append(cmds, cmd)
	}

	return m, tea.Batch(cmds...)
}

func (m SetupWizard) View() string {

	if m.saved {
		return "Settings saved."
	}

	s := "=== General ===\n\n"

	s += m.ticketRepo.Title + "\n"
	for _, t := range m.ticketRepo.Items() {
		cursor := " "
		isFocused := m.ticketRepo.FilterInput.Focused()
		isSelected := t == m.ticketRepo.SelectedItem()
		if isFocused && isSelected {
			cursor = ">"
		} else if isSelected {
			cursor = "-"
		}
		if t, ok := t.(ticketRepo); ok {
			s += fmt.Sprintf("%s %s - %s\n", cursor, t.Title(), t.Description())
		} else {
			s += fmt.Sprintf("%s %s\n", cursor, t.FilterValue())
		}
	}
	s += "\n"

	s += "=== Jira ===\n\n"

	s += m.jiraDomain.View() + "\n"
	s += m.jiraUser.View() + "\n"
	s += m.jiraToken.View() + "\n"
	s += "(note: the token will be stored in plain text on your computer.)\n"

	s += "\n"

	if m.back != nil {
		s += "Press 'esc' to go back.\n"
	} else {
		s += "Press 'esc' to quit.\n"
	}
	s += "Press 'ctrl+s' to save.\n"

	return strings.TrimSpace(s)
}

func (m SetupWizard) focusPrev() (tea.Model, tea.Cmd) {
	if m.jiraDomain.Focused() {
		m.jiraDomain.Blur()
		return m, m.ticketRepo.FilterInput.Focus()

	} else if m.jiraToken.Focused() {
		m.jiraToken.Blur()
		return m, m.jiraUser.Focus()

	} else if m.jiraUser.Focused() {
		m.jiraUser.Blur()
		return m, m.jiraDomain.Focus()

	} else {
		return m.exit()

	}
}

func (m SetupWizard) focusNext() (SetupWizard, tea.Cmd) {
	if m.ticketRepo.FilterInput.Focused() {
		m.ticketRepo.FilterInput.Blur()
		return m, m.jiraDomain.Focus()

	} else if m.jiraDomain.Focused() {
		m.jiraDomain.Blur()
		return m, m.jiraUser.Focus()

	} else if m.jiraUser.Focused() {
		m.jiraUser.Blur()
		return m, m.jiraToken.Focus()

	} else if m.jiraToken.Focused() {
		m.jiraToken.Blur()
		return m, m.ticketRepo.FilterInput.Focus()

	} else {
		return m, nil
	}
}

func (m SetupWizard) exit() (tea.Model, tea.Cmd) {
	if m.back != nil {
		return m.back, func() tea.Msg { return m }

	} else {
		return m, tea.Quit

	}
}

func (m SetupWizard) save() {
	m.Config.SetTicketManagerValue(m.ticketRepo.SelectedItem().FilterValue())
	m.Config.SetJiraCredentials(url.URL{
		Scheme: "https",
		Host:   m.jiraDomain.Value(),
		User: url.UserPassword(
			m.jiraUser.Value(),
			m.jiraToken.Value()),
	})
}

func (m SetupWizard) ticketIndex(v string) int {
	for i, t := range m.ticketRepo.Items() {
		if t.FilterValue() == v {
			return i
		}
	}
	return 0
}

type ticketRepo struct {
	Key         string
	title       string
	description string
}

func (o ticketRepo) Title() string {
	return o.title
}

func (o ticketRepo) Description() string {
	return o.description
}

func (o ticketRepo) FilterValue() string {
	return o.Key
}
