package scene

import (
	"fmt"
	"log"
	"log/slog"
	"strings"
	"time"

	tea "github.com/charmbracelet/bubbletea"
	"github.com/google/uuid"
	"vnbr.de/track/internal/config"
	"vnbr.de/track/internal/export"
	"vnbr.de/track/internal/ticket"
	"vnbr.de/track/internal/timer"
	"vnbr.de/track/internal/util"
	"vnbr.de/track/internal/view/widget"
)

type Timer struct {
	Config     config.Config
	TimerRepo  timer.Repository
	TicketRepo ticket.Manager

	showHistory bool

	all       []uuid.UUID
	curr      uuid.UUID
	err       error
	submitted bool
}

func NewTimer(config config.Config) Timer {
	repo := config.InitTimerRepositoryOrQuit()
	mgr := config.NewTicketManager()
	timers, err := repo.ListIds()
	if err != nil {
		log.Fatal(err)
	}

	var timer uuid.UUID
	if len(timers) > 0 {
		timer = timers[0]
	}

	return Timer{
		Config:     config,
		TimerRepo:  repo,
		TicketRepo: mgr,

		all:  timers,
		curr: timer,
	}
}

func (m Timer) Init() tea.Cmd {
	return tick()
}

func (m Timer) Update(msg tea.Msg) (tea.Model, tea.Cmd) {
	var cmds []tea.Cmd

	switch msg := msg.(type) {

	case tickMsg:
		return m, tick()

	case SetupWizard:
		cmds = append(cmds, tick())

	case TicketAssignmentMsg:
		cmds = append(cmds, tick())
		switch msg.Err {
		case nil:
			if err := m.TimerRepo.Assign(m.curr, msg.Ticket); err != nil {
				m.err = err
			}
		case AssignmentAborted:
			// noop
		default:
			m.err = msg.Err
			slog.Error("ticket assignment", "error", m.err)
		}

	case TicketCreated:
		m.err = msg.Err
		cmds = append(cmds, tick())

	case widget.DurationPromptMsg:
		cmds = append(cmds, tick())
		switch msg.Error {
		case nil:
			if r, ok := m.TimerRepo.(timer.EditableRepository); ok {
				r.SetDuration(m.curr, msg.Duration)
			} else {
				m.err = fmt.Errorf("Timer cannot be edited")
			}
		case widget.Cancelled:
		// noop
		default:
			m.err = msg.Error
			slog.Error("duration assignment", "error", m.err)
		}

	case widget.PromptMsg:
		cmds = append(cmds, tick())
		switch msg.Error {
		case nil:
			m.TimerRepo.Comment(m.curr, msg.Input)
		case widget.Cancelled:
		// noop
		default:
			m.err = msg.Error
			slog.Error("comment", "error", m.err)
		}

	case tea.KeyMsg:
		switch msg.String() {

		case ".":
			scene := NewSetupWizard(m.Config, m)
			return scene, scene.Init()

		case "+":
			m.all = append(m.all, uuid.New())

		case "a":
			scene := NewTicketAssignment(m.TicketRepo).Return(m)
			return scene, scene.Init()

		case "c":
			scene := widget.NewPrompt("Comment", m.TimerRepo.GetComment(m.curr))
			scene.Back = m
			return scene, scene.Init()

		case "e":
			if _, ok := m.TimerRepo.(timer.EditableRepository); ok {
				if durr, err := m.TimerRepo.GetDuration(m.curr); err == nil {
					scene := widget.NewDurationPrompt("Duration", durr)
					scene.Back = m
					return scene, scene.Init()
				} else {
					m.err = err
				}
			}

		case "h":
			m.showHistory = !m.showHistory

		case "j", "down":
			for idx := range m.all {
				if m.curr == m.all[idx] && len(m.all) > (idx+1) {
					m.curr = m.all[idx+1]
					m.err = nil
					m.submitted = false
					break
				}
			}

		case "k", "up":
			for idx := range m.all {
				if m.all[idx] == m.curr && idx > 0 {
					m.curr = m.all[idx-1]
					m.err = nil
					m.submitted = false
					break
				}
			}

		case "n":
			_, canAppend := m.TicketRepo.(ticket.AppendManager)
			_, canCreate := m.TicketRepo.(ticket.CreateManager)
			if canAppend || canCreate {
				scene := NewTicketCreation(m.TicketRepo, WithReturn(m))
				return scene, scene.Init()
			}

		case "p":
			if m.isPublishEnabled() {
				if err := m.publish(); err != nil {
					m.err = err
				} else {
					m.submitted = true
					return m, tea.Quit
				}
			}

		case "r":
			if err := m.TimerRepo.Reset(m.curr); err != nil {
				m.err = err
				slog.Error("timer reset", "error", m.err)
			}

		case "s":
			if err := m.TimerRepo.Toggle(m.curr); err != nil {
				m.err = err
				slog.Error("timer toggle", "error", m.err)
			}

		case "q":
			return m, tea.Quit

		}
	}

	return m, tea.Batch(cmds...)
}

func (m Timer) View() string {
	_, canAppend := m.TicketRepo.(ticket.AppendManager)
	_, canCreate := m.TicketRepo.(ticket.CreateManager)
	_, canEdit := m.TimerRepo.(timer.EditableRepository)
	_, canTrace := m.TimerRepo.(timer.TraceableRepository)

	s := ""

	if m.err != nil {
		s += "Error: " + m.err.Error() + "\n"
	}

	s += "\n"

	runs := m.TimerRepo.GetState(m.curr) == timer.Running

	if m.showHistory {
		if r, ok := m.TimerRepo.(timer.TraceableRepository); ok {
			hist, err := r.GetEvents(m.curr)
			if err != nil {
				m.err = err
			}
			for _, e := range hist {
				t := e.Time.Format("2006-01-02 15:04:05")
				switch e.EventType {
				case timer.Start:
					if e.Duration == time.Duration(0) {
						s += fmt.Sprintf("%s: started\n", t)
					} else {
						s += fmt.Sprintf("%s: resumed at %s\n", t, e.Duration.String())
					}
				case timer.Stop:
					s += fmt.Sprintf("%s: stopped at %s\n", t, e.Duration.String())
				case timer.Edit:
					s += fmt.Sprintf("%s: changed to %s\n", t, e.Duration.String())
				case timer.Reset:
					s += fmt.Sprintf("%s: reset to 0m\n", t)
				case timer.Assign:
					s += fmt.Sprintf("%s: assigned to %s: %s\n", t, e.Ticket.Key, e.Ticket.Title)
				case timer.Comment:
					s += fmt.Sprintf("%s: comment: %s\n", t, e.Comment)
				}
			}
			s += "\n"
		}
	}

	var ts strings.Builder
	ts.WriteString(s)
	ts.WriteString("Available Timers:\n")

	for _, ti := range m.all {
		ticket, _ := m.TimerRepo.Get(ti)
		title := util.FirstValue(ticket.Ticket.Title, "Unasigned Ticket")

		fmt.Fprintf(&ts, "%s [%s %s] %s\n",
			util.When(ti == m.curr, ">", " "),
			ticket.GetStateIcon(),
			ticket.GetElapsed(),
			util.When(ticket.Comment != "", fmt.Sprintf("%s: %s", title, ticket.Comment), title),
		)
	}

	if m.submitted {
		ts.WriteString("\nTimer submitted\n")
	}

	s = ts.String()+"\n"

	s += "Press 'a' to assign a ticket.\n"
	s += "Press 'c' to add a comment.\n"
	if canEdit {
		s += "Press 'e' to edit/set the duration.\n"
	}
	if canTrace {
		s += "Press 'h' to toggle timer history.\n"
	}
	if canAppend || canCreate {
		s += "Press 'n' to create new ticket.\n"
	}
	if m.isPublishEnabled() {
		s += "Press 'p' to publish and reset the timer.\n"
	}
	s += "Press 'r' to reset the timer.\n"
	if runs {
		s += "Press 's' to stop the timer.\n"
	} else {
		s += "Press 's' to start the timer.\n"
	}
	s += "Press 'q' to quit.\n"

	return strings.TrimSpace(s)
}

type tickMsg struct{}

func (m Timer) isPublishEnabled() bool {
	 // Jira not enabled, but the only supported backend.
	return m.Config.GetTicketManagerValue() == "jira"
}

func (m Timer) publish() error {
	tm, err := m.TimerRepo.Get(m.curr)
	if err != nil {
		return err
	}

	cw := export.NewClockwork(m.Config.GetJiraCredentials())
	err = cw.Publish(time.Now().Add(-tm.GetElapsed()), tm)

	// Reset timer to not submit it twice.
	// TODO: This probably should show up in the history.
	m.TimerRepo.Reset(m.curr)
	return err
}

func tick() tea.Cmd {
	return tea.Tick(time.Second, func(t time.Time) tea.Msg { return tickMsg{} })
}
