package scene

import (
	"log/slog"

	"github.com/charmbracelet/bubbles/textinput"
	tea "github.com/charmbracelet/bubbletea"
	"vnbr.de/track/internal/ticket"
	"vnbr.de/track/internal/view/widget"
)

// Msg emitted when a ticket was either created or the user cancelled it.
type TicketCreated struct {
	Ticket ticket.Ticket
	Err    error
}

type TicketCreationOpt func(*TicketCreation)

type focused int

const (
	focusOnKey focused = iota
	focusOnTitle
	focusOnConfirm
	focusOnSubmit
)

type TicketCreation struct {
	//dependencies
	repo ticket.Manager

	//state
	back  tea.Model
	curr  focused
	err   error
	focus bool

	// ui components
	key    textinput.Model
	title  textinput.Model
	submit bool
}

func NewTicketCreation(mgr ticket.Manager, opts ...TicketCreationOpt) *TicketCreation {
	m := &TicketCreation{
		repo: mgr,

		key:   textinput.New(),
		title: textinput.New(),
	}

	m.key.Prompt = "ID: "
	m.title.Prompt = "Title: "

	switch m.repo.(type) {
	case ticket.AppendManager:
		m.curr = focusOnTitle
		m.title.Focus()
	case ticket.CreateManager:
		m.key.Focus()
	}

	for _, opt := range opts {
		opt(m)
	}

	return m
}

func WithReturn(back tea.Model) TicketCreationOpt {
	return func(m *TicketCreation) {
		m.back = back
	}
}

func (m TicketCreation) Init() tea.Cmd {
	switch m.repo.(type) {
	case ticket.AppendManager:
		return m.title.Focus()
	case ticket.CreateManager:
		return m.key.Focus()
	}

	return nil
}

func (m TicketCreation) Update(msg tea.Msg) (tea.Model, tea.Cmd) {
	var cmd tea.Cmd
	var cmds []tea.Cmd

	m.key, cmd = m.key.Update(msg)
	cmds = append(cmds, cmd)

	m.title, cmd = m.title.Update(msg)
	cmds = append(cmds, cmd)

	switch msg := msg.(type) {

	case TicketCreated:
		return m.quit(msg.Ticket, msg.Err)

	case tea.KeyMsg:
		switch msg.String() {

		case "shift+tab":
			fallthrough

		case "esc":
			_, canCreate := m.repo.(ticket.CreateManager)
			switch m.curr {
			case focusOnKey:
				return m.quit(ticket.Ticket{}, widget.Cancelled)
			case focusOnTitle:
				if !canCreate {
					return m.quit(ticket.Ticket{}, widget.Cancelled)
				}
				m.title.Blur()
				cmds = append(cmds, m.key.Focus())
			case focusOnConfirm:
				cmds = append(cmds, m.title.Focus())
			}

			m.curr--

		case "tab":
			fallthrough

		case "enter":
			switch m.curr {
			case focusOnKey:
				if m.key.Value() != "" {
					m.curr++
					m.key.Blur()
					cmds = append(cmds, m.title.Focus())
				}
			case focusOnTitle:
				if m.title.Value() != "" {
					m.curr++
					m.title.Blur()
				}
			case focusOnConfirm:
				m.curr++
				if m.submit {
					cmds = append(cmds, m.create())
				} else {
					return m.quit(ticket.Ticket{}, widget.Cancelled)
				}
			}

		case "left", "y", "h", "j":
			if m.curr == focusOnConfirm {
				m.submit = true
			}

		case "right", "n", "l", "k":
			if m.curr == focusOnConfirm {
				m.submit = false
			}
		}
	}

	return m, tea.Batch(cmds...)
}

func (m TicketCreation) View() string {
	_, canCreate := m.repo.(ticket.CreateManager)

	s := ""

	switch m.curr {
	case focusOnKey:
		fallthrough

	case focusOnTitle:
		if canCreate {
			s += m.key.View() + "\n"
		}
		s += m.title.View() + "\n"

	case focusOnConfirm:
		if canCreate {
			s += m.key.Value() + ": " + m.title.Value() + "\n\n"
		} else {
			s += m.title.Value() + "\n\n"
		}
		s += "Create ticket?\n\n"
		if m.submit {
			s += "[x] [Y]es    [ ] [N]o\n"
		} else {
			s += "[ ] [Y]es    [x] [N]o\n"
		}

	case focusOnSubmit:
		s += "Creating ticket..."
	}

	return s
}

func (m TicketCreation) create() tea.Cmd {
	return func() tea.Msg {
		t := ticket.Ticket{Key: m.key.Value(), Title: m.title.Value()}
		switch repo := m.repo.(type) {

		case ticket.AppendManager:
			slog.Debug("Add ticket", "key", t.Key, "title", t.Title)
			t, err := repo.Add(t.Title)
			return TicketCreated{t, err}

		case ticket.CreateManager:
			slog.Debug("Add ticket", "key", t.Key, "title", t.Title)
			t, err := repo.Add(t)
			return TicketCreated{t, err}

		default:
			return TicketCreated{t, widget.Cancelled}
		}
	}
}

func (m TicketCreation) quit(t ticket.Ticket, err error) (tea.Model, tea.Cmd) {
	if m.back != nil {
		return m.back, func() tea.Msg {
			return TicketCreated{t, err}
		}
	} else {
		return m, tea.Quit
	}
}
