package scene

import (
	"fmt"

	tea "github.com/charmbracelet/bubbletea"
	"vnbr.de/track/internal/ticket"
	"vnbr.de/track/internal/view/widget"
)

var AssignmentAborted = fmt.Errorf("Ticket selection aborted by user")

type focus int

const (
	base focus = iota
	search
)

type TicketAssignmentMsg struct {
	Ticket ticket.Ticket
	Err    error
}

type TicketAssignment struct {
	TicketRepo ticket.Manager
	Back       tea.Model

	current int
	err     error
	focus   focus
	loaded bool
	tickets []ticket.Ticket

	searchPrompt widget.Prompt
}

func NewTicketAssignment(mgr ticket.Manager) *TicketAssignment {
	return &TicketAssignment{
		TicketRepo: mgr,
		Back:       nil,
	}
}

func (m *TicketAssignment) Return(back tea.Model) *TicketAssignment {
	m.Back = back
	return m
}

func (m *TicketAssignment) Selected() (ticket.Ticket, bool) {
	if m.current < len(m.tickets) {
		return m.tickets[m.current], true
	} else {
		return ticket.Ticket{}, false
	}
}

func (m TicketAssignment) Init() tea.Cmd {
	return m.search()
}

func (m TicketAssignment) Update(msg tea.Msg) (tea.Model, tea.Cmd) {
	switch msg := msg.(type) {

	case errMsg:
		m.err = msg

	case searchMsg:
		m.current = 0
		m.loaded = true
		m.tickets = msg

	case tea.KeyMsg:
		switch msg.String() {

		case "esc", "q":
			return m.quit(AssignmentAborted)

		case "enter":
			return m.quit(nil)

		case "up", "ctrl+p":
			if m.current > 0 {
				m.current--
			}
			return m, nil

		case "down", "ctrl+n":
			if m.current+1 < len(m.tickets) {
				m.current++
			}
			return m, nil

		}
	}

	mod, cmd := m.searchPrompt.Update(msg)
	if mod, ok := mod.(widget.Prompt); ok {
		if mod.Value != m.searchPrompt.Value {
			m.searchPrompt = mod
			m.loaded = false
			return m, tea.Batch(cmd, m.search())
		}
	}

	return m, nil
}

func (m TicketAssignment) View() string {
	var c string
	s := fmt.Sprintf("Select a ticket: %s\n", m.searchPrompt.Value)

	if m.err != nil {
		s += fmt.Sprintf("ERROR: %s\n", m.err)
	}

	s += "\n"

	if ! m.loaded {
		s += "fetching tickets...\n"
		return s
	}

	for i, t := range m.tickets {
		if i == int(m.current) {
			c = ">"
		} else {
			c = " "
		}

		s += fmt.Sprintf("%s %s: %s\n", c, t.Key, t.Title)
	}

	return s
}

type errMsg error
type searchMsg []ticket.Ticket

func (m *TicketAssignment) search() tea.Cmd {
	return func() tea.Msg {
		if tickets, err := m.TicketRepo.Search(m.searchPrompt.Value); err == nil {
			return searchMsg(tickets)
		} else {
			return errMsg(err)
		}

	}
}

func (m TicketAssignment) quit(err error) (tea.Model, tea.Cmd) {
	if m.Back != nil {
		t, ok := m.Selected()
		return m.Back, func() tea.Msg {
			if err == nil && !ok {
				err = AssignmentAborted
			}
			return TicketAssignmentMsg{t, err}
		}
	} else {
		return m, tea.Quit
	}
}
