package timer

import (
	"database/sql"
	"embed"
	"fmt"
	"log/slog"
	"os"
	"path"
	"strconv"
	"time"

	"github.com/golang-migrate/migrate/v4"
	"github.com/golang-migrate/migrate/v4/database/sqlite3"
	"github.com/golang-migrate/migrate/v4/source/iofs"
	"github.com/google/uuid"
	_ "github.com/mattn/go-sqlite3"
	"vnbr.de/track/internal/fs"
	"vnbr.de/track/internal/ticket"
)

//go:embed sqlite/migrations/*.sql
var migrationsFS embed.FS

const migrationsDir = "sqlite/migrations"

type SqliteRepository struct {
	db *sql.DB
}

func NewSqliteRepository() (r *SqliteRepository, err error) {
	stateDir, err := fs.UserStateDir()
	if err != nil {
		return
	}

	dbFile := path.Join(stateDir, "dallytrack", "track.db")
	if err = os.MkdirAll(path.Dir(dbFile), 0700); err != nil {
		return
	}

	db, err := sql.Open("sqlite3", dbFile)
	if err != nil {
		return
	}

	r = &SqliteRepository{db}
	err = r.migrate()
	return
}

func (r *SqliteRepository) Reset(id uuid.UUID) error {
	prev := r.getTimer(id)
	if prev.State == Running {
		// Stop timer to add full elapsed time to history.
		r.Toggle(id)
	}
	e := Event{time.Now(), Reset, Timer{
		Ticket:  prev.Ticket,
		Comment: prev.Comment,
	}}
	return r.store(id, e)
}

func (r *SqliteRepository) Comment(id uuid.UUID, c string) error {
	e := Event{time.Now(), Comment, r.getTimer(id)}
	e.Timer.Comment = c
	return r.store(id, e)
}

func (r *SqliteRepository) Assign(id uuid.UUID, t ticket.Ticket) error {
	e := Event{time.Now(), Assign, r.getTimer(id)}
	e.Timer.Ticket = t
	return r.store(id, e)
}

func (r *SqliteRepository) Toggle(id uuid.UUID) error {
	t := r.getTimer(id)
	switch t.State {
	case Running:
		e := Event{time.Now(), Stop, t}
		e.Timer.Duration = e.Timer.GetElapsed()
		e.Timer.StartedAt = time.Time{}
		e.Timer.State = Stopped
		return r.store(id, e)
	case Stopped:
		e := Event{time.Now(), Start, t}
		e.Timer.StartedAt = time.Now()
		e.Timer.State = Running
		return r.store(id, e)
	default:
		return fmt.Errorf("Unsupported timer state: " + strconv.Itoa(int(t.State)))
	}
}

func (r *SqliteRepository) GetAssigned(id uuid.UUID) ticket.Ticket {
	return r.getTimer(id).Ticket
}

func (r *SqliteRepository) GetComment(id uuid.UUID) string {
	return r.getTimer(id).Comment
}

func (r *SqliteRepository) GetDuration(id uuid.UUID) (duration time.Duration, err error) {
	return r.getTimer(id).GetElapsed(), nil
}

func (r *SqliteRepository) GetEvents(id uuid.UUID) ([]Event, error) {
	stmt, err := r.db.Prepare(`
		SELECT event_type, created_at, started_at, duration, state, comment, ticket_id, ticket_title
		FROM timers
		WHERE uuid = ?
		ORDER BY id ASC;
		`)
	if err != nil {
		return []Event{}, err
	}

	data, err := stmt.Query(id)
	if err != nil {
		return []Event{}, err
	}

	list, err := readEvents(data)
	if err != nil {
		return []Event{}, err
	}

	return list, nil
}

func (r *SqliteRepository) GetState(id uuid.UUID) State {
	return r.getTimer(id).State
}

func (r *SqliteRepository) ListIds() (ids uuid.UUIDs, err error) {
	stmt, err := r.db.Prepare(`SELECT DISTINCT uuid FROM timers ORDER BY created_at DESC;`)
	if err != nil {
		return
	}

	data, err := stmt.Query()
	if err != nil {
		return
	}

	for data.Next() {
		if err = data.Err(); err != nil {
			return
		}

		var id uuid.UUID
		if err = data.Scan(&id); err != nil {
			return
		}

		ids = append(ids, id)
	}

	return
}

func (r *SqliteRepository) SetDuration(id uuid.UUID, to time.Duration) error {
	e := Event{time.Now(), Edit, r.getTimer(id)}

	e.Timer.Duration = to
	if e.Timer.State == Running {
		e.Timer.StartedAt = time.Now()
	}

	return r.store(id, e)
}

func (r *SqliteRepository) Get(id uuid.UUID) (t Timer, err error) {
	t = Timer{}

	stmt, err := r.db.Prepare(`
		SELECT event_type, created_at, started_at, duration, state, comment, ticket_id, ticket_title
		FROM timers
		WHERE uuid = ?
		ORDER BY id DESC
		LIMIT 1;
		`)
	if err != nil {
		return
	}

	data, err := stmt.Query(id)
	if err != nil {
		return
	}

	list, err := readEvents(data)
	if err != nil {
		return
	}

	if len(list) > 0 {
		return list[len(list)-1].Timer, err
	}

	return
}

func (r *SqliteRepository) getTimer(id uuid.UUID) Timer {
	t, _ := r.Get(id)
	return t
}

func (r *SqliteRepository) migrate() error {
	dir, err := iofs.New(migrationsFS, migrationsDir)
	if err != nil {
		return err
	}
	driver, err := sqlite3.WithInstance(r.db, &sqlite3.Config{})
	if err != nil {
		return err
	}
	m, err := migrate.NewWithInstance("iofs", dir, "sqlite3", driver)
	if err != nil {
		return err
	}
	switch err = m.Up(); err {
	case migrate.ErrNoChange:
	// noop
	default:
		return err
	}

	return nil
}

func (r *SqliteRepository) store(id uuid.UUID, e Event) error {
	slog.Debug("timer: "+e.EventType.String(), "data", e.Timer, "type", "sqlite", "id", id)

	stmt, err := r.db.Prepare(`
		INSERT INTO timers
		(uuid, event_type, created_at, started_at, duration, state, comment, ticket_id, ticket_title)
		VALUES (?,?,?,?,?,?,?,?,?);
		`)

	if err != nil {
		return err
	}

	if _, err := stmt.Exec(
		id,
		int(e.EventType),
		time.Now().Unix(),
		e.StartedAt.Unix(),
		e.Duration.Round(time.Second),
		int(e.State),
		e.Comment,
		e.Ticket.Key,
		e.Ticket.Title,
	); err != nil {
		return err
	}

	return nil
}

func readEvents(rows *sql.Rows) ([]Event, error) {
	list := []Event{}
	type t int

	for rows.Next() {
		if err := rows.Err(); err != nil {
			return list, err
		}

		var et int
		var state int

		e := Event{}
		err := rows.Scan(
			&et,
			&e.Time,
			&e.StartedAt,
			&e.Duration,
			&state,
			&e.Comment,
			&e.Ticket.Key,
			&e.Ticket.Title,
		)
		e.EventType = EventType(et)
		e.State = State(state)

		if err != nil {
			return list, err
		}

		list = append(list, e)
	}

	return list, nil
}
