package timer

import (
	"log/slog"
	"time"

	"github.com/google/uuid"
	"vnbr.de/track/internal/ticket"
)

type MemoryRepository struct {
	events map[uuid.UUID][]Event
}

func NewMemoryRepository() *MemoryRepository {
	return &MemoryRepository{
		events: make(map[uuid.UUID][]Event),
	}
}

func (r *MemoryRepository) Reset(id uuid.UUID) error {
	prev := r.getTimer(id)
	t := Timer{
		Ticket:  prev.Ticket,
		Comment: prev.Comment,
	}

	slog.Debug("timer: reset", "data", t, "type", "memory", "id", id)
	r.events[id] = append(r.events[id], Event{time.Now(), Reset, t})
	return nil
}

func (r *MemoryRepository) Comment(id uuid.UUID, c string) error {
	t := r.getTimer(id)
	t.Comment = c
	slog.Debug("timer: comment", "data", t, "type", "memory", "id", id)
	r.events[id] = append(r.events[id], Event{time.Now(), Comment, t})
	return nil
}

func (r *MemoryRepository) Assign(id uuid.UUID, tk ticket.Ticket) error {
	tm := r.getTimer(id)
	tm.Ticket = tk
	slog.Debug("timer: assign", "data", tm, "type", "memory", "id", id)
	r.events[id] = append(r.events[id], Event{time.Now(), Assign, tm})
	return nil
}

func (r *MemoryRepository) Toggle(id uuid.UUID) error {
	t := r.getTimer(id)
	switch r.GetState(id) {
	case Running:
		t.Duration = t.GetElapsed()
		t.StartedAt = time.Time{}
		t.State = Stopped
		slog.Debug("timer: stop", "data", t, "type", "memory", "id", id)
		r.events[id] = append(r.events[id], Event{time.Now(), Stop, t})
	case Stopped:
		t.StartedAt = time.Now()
		t.State = Running
		slog.Debug("timer: start", "data", t, "type", "memory", "id", id)
		r.events[id] = append(r.events[id], Event{time.Now(), Start, t})
	}
	return nil
}

func (r *MemoryRepository) Get(id uuid.UUID) (Timer, error) {
	return r.getTimer(id), nil
}

func (r *MemoryRepository) GetAssigned(id uuid.UUID) ticket.Ticket {
	return r.getTimer(id).Ticket
}

func (r *MemoryRepository) GetComment(id uuid.UUID) string {
	return r.getTimer(id).Comment
}

func (r *MemoryRepository) GetDuration(id uuid.UUID) (duration time.Duration, err error) {
	return r.getTimer(id).GetElapsed(), nil
}

func (r *MemoryRepository) GetEvents(id uuid.UUID) ([]Event, error) {
	return r.events[id], nil
}

func (r *MemoryRepository) GetState(id uuid.UUID) State {
	return r.getTimer(id).State
}

func (r *MemoryRepository) ListIds() (uuid.UUIDs, error) {
	var keys uuid.UUIDs
	for key, _ := range r.events {
		keys = append(keys, key)
	}
	return keys, nil
}

func (r *MemoryRepository) SetDuration(id uuid.UUID, to time.Duration) error {
	t := r.getTimer(id)

	t.Duration = to
	if t.State == Running {
		t.StartedAt = time.Now()
	}

	r.events[id] = append(r.events[id], Event{time.Now(), Edit, t})

	slog.Debug("timer: edit", "data", t, "type", "memory", "id", id)
	return nil
}

func (r *MemoryRepository) getTimer(id uuid.UUID) Timer {
	list, _ := r.GetEvents(id)

	if len(list) > 0 {
		return list[len(list)-1].Timer
	}

	return Timer{}
}
