package timer

import (
	"time"

	"github.com/google/uuid"
	"vnbr.de/track/internal/ticket"
)

const (
	Start EventType = iota
	Stop
	Edit
	Reset
	Assign
	Comment
)

const (
	Stopped State = iota
	Running
)

type EventType int

func (e EventType) String() string {
	switch e {
	case Start:
		return "start"
	case Stop:
		return "stop"
	case Edit:
		return "edit"
	case Reset:
		return "reset"
	case Assign:
		return "assign"
	case Comment:
		return "comment"
	default:
		return "unknown"
	}
}

type Event struct {
	time.Time
	EventType
	Timer
}

type State int

type Repository interface {
	Assign(uuid.UUID, ticket.Ticket) error
	Comment(uuid.UUID, string) error
	Reset(uuid.UUID) error
	Toggle(uuid.UUID) error

	Get(uuid.UUID) (Timer, error)
	GetAssigned(uuid.UUID) ticket.Ticket
	GetComment(uuid.UUID) string
	GetDuration(uuid.UUID) (time.Duration, error)
	GetState(uuid.UUID) State

	ListIds() (uuid.UUIDs, error)
}

type EditableRepository interface {
	Repository
	SetDuration(uuid.UUID, time.Duration) error
}

type TraceableRepository interface {
	Repository
	GetEvents(uuid.UUID) ([]Event, error)
}

type Timer struct {
	Duration  time.Duration
	StartedAt time.Time
	State     State

	Comment string
	Ticket  ticket.Ticket
}

func (t Timer) GetElapsed() time.Duration {
	e := t.Duration

	if t.State == Running {
		e += time.Now().Sub(t.StartedAt)
	}

	return e.Round(time.Second)
}

func (t Timer) GetStateIcon() string {
	switch t.State {
	case Running:
		return "\u23fa" // ⏺︎
	case Stopped:
		return "\u23f8" // ⏸︎
	default:
		return "\u23f9" // ⏹︎
	}
}
