package ticket

import (
	"encoding/json"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strings"
	"time"
)

type jiraErrorResponse struct {
	ErrorMessages []string `json:"errorMessages"`
}

type jiraIssueFields struct {
	Summary string `json:"summary"`
}

type jiraIssueResponse struct {
	Key   string `json:"key"`
	Title string `json:"fields.summary"`

	jiraIssueFields `json:"fields"`
}

type jiraIssuePickerResponse struct {
	Sections []struct {
		Issues []struct {
			Key string `json:"key"`
			// Issue summary with the match wrapped in <b>...</b> tags.
			SummaryHtml string `json:"summary"`
			SummaryText string `json:"summaryText"`
		} `json:"issues"`
	} `json:"sections"`
}

type Jira struct {
	url url.URL
}

func NewJiraTicketRepository(u url.URL) Jira {
	return Jira{u}
}

func (m *Jira) Get(key string) (Ticket, error) {
	key = strings.ToUpper(key)
	allowedRunes := "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890-"

	for _, c := range key {
		if !strings.Contains(allowedRunes, string(c)) {
			return Ticket{Key: key}, fmt.Errorf("Invalid issue key given.")
		}
	}

	ticket := jiraIssueResponse{}
	if err := req(m.url.JoinPath("/rest/api/3/issue/", key), &ticket); err != nil {
		return Ticket{Key: key}, err
	}

	return Ticket{
		Key:   ticket.Key,
		Title: ticket.Summary,
	}, nil
}

func (m *Jira) Search(query string) ([]Ticket, error) {
	picks := jiraIssuePickerResponse{}

	u := m.url.JoinPath("/rest/api/3/issue/picker")
	u.RawQuery = "query=" + query
	if err := req(u, &picks); err != nil {
		return []Ticket{}, err
	}

	tickets := []Ticket{}
	for _, pick := range picks.Sections[0].Issues {
		tickets = append(tickets, Ticket{
			Key:   pick.Key,
			Title: pick.SummaryText,
		})
	}

	return tickets, nil
}

func req(u *url.URL, v any) error {
	username := u.User.Username()
	password, _ := u.User.Password()

	client := http.Client{Timeout: time.Duration(15 * time.Second)}
	req := &http.Request{
		Method: "GET",
		URL:    u,
		Header: http.Header{}, // without this, .SetBasicAuth panics
	}
	req.SetBasicAuth(username, password)

	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	body, _ := io.ReadAll(resp.Body)

	jer := jiraErrorResponse{}
	err = json.Unmarshal(body, &jer)
	if len(jer.ErrorMessages) > 0 {
		return fmt.Errorf(jer.ErrorMessages[0])
	}

	err = json.Unmarshal(body, v)
	if err != nil {
		return err
	}

	if resp.StatusCode != 200 {
		slog.Debug("jira ticket", "response", resp, "body", body)
		return fmt.Errorf("Something went wrong")
	}

	return nil
}
