package ticket

import (
	"strings"

	"gopkg.in/ini.v1"
)

type FileManager struct {
	file    *ini.File
	path    string
	section string
}

func NewFileManager(path string) FileManager {
	file, err := ini.Load(path)

	if err != nil {
		// There is no point in instantiating this manager if we cannot read
		// the config file containing the tickets. The initiator has to ensure,
		// that the file actually exists and is readable.
		panic(err)
	}

	return FileManager{
		file:    file,
		path:    path,
		section: "tickets",
	}
}

func (m *FileManager) Add(t Ticket) (Ticket, error) {
	m.file.Section(m.section).Key(t.Key).SetValue(t.Title)
	m.file.SaveTo(m.path)
	return t, nil
}

func (m *FileManager) Get(key string) (Ticket, error) {
	kv := m.file.Section(m.section).Key(key)

	if kv == nil {
		return Ticket{}, NotFound
	}

	return Ticket{
		Key:   key,
		Title: kv.Value(),
	}, nil
}

func (m *FileManager) Search(query string) ([]Ticket, error) {
	tickets := []Ticket{}
	kv := m.file.Section(m.section).KeysHash()

	for k, v := range kv {
		if query == "" || strings.Contains(k, query) || strings.Contains(v, query) {
			tickets = append(tickets, Ticket{Key: k, Title: v})
		}
	}

	return tickets, nil
}

func (m *FileManager) WithSection(s string) func(*FileManager) {
	return func(fm *FileManager) { m.section = s }
}
