package export

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strings"
	"time"

	"vnbr.de/track/internal/timer"
)

func NewClockwork(url url.URL) Clockwork {
	return Clockwork{url}
}

type Clockwork struct {
	url url.URL
}

type jiraErrorResponse struct {
	ErrorMessages []string `json:"errorMessages"`
}

type jiraWorklogCreateResponse struct {
	Self      url.URL       `json:"self"`
	Created   time.Time     `json:"created"`
	Updated   time.Time     `json:"updated"`
	Started   time.Time     `json:"Started"`
	TimeSpent time.Duration `json:"timeSpentSeconds"`
}

func (m *Clockwork) Publish(start time.Time, tm timer.Timer) error {
	key := strings.ToUpper(tm.Ticket.Key)
	var v any
	return create(
		m.url.JoinPath("/rest/api/3/issue", key, "worklog"),
		fmt.Sprintf(`
			{
				"comment": {
					"content": [
						{
						"content": [
							{
							"text": "%s",
							"type": "text"
							}
						],
						"type": "paragraph"
						}
					],
					"type": "doc",
					"version": 1
				},
				"properties": [
					{
						"key": "clockwork",
						"value": {}
					}
				],
				"started": "%s",
				"timeSpentSeconds": %d
			}`,
			tm.Comment,
			start.Format("2006-01-02T15:04:05.000+0700"),
			int(tm.GetElapsed().Round(time.Second).Seconds()),
		),
		&v,
	)
}

func create(u *url.URL, jsonData string, val any) error {
	username := u.User.Username()
	password, _ := u.User.Password()

	client := http.Client{Timeout: time.Duration(15 * time.Second)}
	req, _ := http.NewRequest("POST", u.String(), bytes.NewBuffer([]byte(jsonData)))
	req.Header.Add("Content-Type", "application/json")
	req.SetBasicAuth(username, password)

	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	body, _ := io.ReadAll(resp.Body)

	jer := jiraErrorResponse{}
	err = json.Unmarshal(body, &jer)
	if len(jer.ErrorMessages) > 0 {
		return fmt.Errorf(jer.ErrorMessages[0])
	}

	err = json.Unmarshal(body, val)
	if err != nil {
		return err
	}

	if resp.StatusCode != 201 {
		slog.Debug("jira ticket", "response", resp, "body", body)
		return fmt.Errorf("Something went wrong")
	}

	return nil
}
