package config

import (
	"log/slog"
	"net/url"
	"os"
	"path"

	"gopkg.in/ini.v1"
	"vnbr.de/track/internal/ticket"
	"vnbr.de/track/internal/timer"
)

type Config struct {
	configFile  *ini.File
	configPath  string
	secretsFile *ini.File
	secretsPath string
}

func LoadConfig() Config {
	var err error

	c := Config{}
	c.configPath = getOrInitPath("track.cfg")
	c.configFile, err = ini.Load(c.configPath)

	if err != nil {
		slog.Error("failed loading config", "error", err)
		os.Exit(1)
	}

	c.configFile.SaveTo(c.configPath)

	// make sure we can load auth.cfg
	c.secretsPath = getOrInitPath("auth.cfg")
	if c.secretsFile, err = ini.Load(c.secretsPath); err != nil {
		slog.Error("failed loading config", "error", err)
		os.Exit(1)
	}

	return c
}

func (c *Config) GetJiraCredentials() url.URL {
	return url.URL{
		Scheme: "https",
		User: url.UserPassword(
			c.secretsFile.Section("jira").Key("user").String(),
			c.secretsFile.Section("jira").Key("token").String(),
		),
		Host: c.secretsFile.Section("jira").Key("domain").String(),
		Path: "/",
	}
}

func (c *Config) GetTicketManagerValue() string {
	return c.configFile.Section("settings").Key("tickets").String()
}

func (c *Config) InitTimerRepository() (timer.Repository, error) {
	return timer.NewSqliteRepository()
}

func (c *Config) InitTimerRepositoryOrQuit() timer.Repository {
	t, err := c.InitTimerRepository()
	if err != nil {
		slog.Error(err.Error())
		os.Exit(1)
	}
	return t
}

func (c *Config) NewTicketManager() ticket.Manager {
	switch c.GetTicketManagerValue() {

	case "jira":
		mgr := ticket.NewJiraTicketRepository(c.GetJiraCredentials())
		return &mgr

	default:
		mgr := ticket.NewFileManager(c.configPath)
		return &mgr

	}
}

func (c *Config) SetJiraCredentials(u url.URL) {
	pass, _ := u.User.Password()
	c.secretsFile.Section("jira").Key("domain").SetValue(u.Host)
	c.secretsFile.Section("jira").Key("user").SetValue(u.User.Username())
	c.secretsFile.Section("jira").Key("token").SetValue(pass)
	c.secretsFile.SaveTo(c.secretsPath)
}

func (c *Config) SetTicketManagerValue(v string) {
	c.configFile.Section("settings").Key("tickets").SetValue(v)
	c.configFile.SaveTo(c.configPath)
}

func getOrInitPath(filename string) string {
	userConfDir, err := os.UserConfigDir()

	if err != nil {
		slog.Error("failed loading config", "error", err)
		os.Exit(1)
	}

	// path where the config should be
	confPath := path.Join(userConfDir, "dallytrack", filename)

	// make sure the directory exists
	confDir := path.Dir(confPath)
	if _, err := os.Stat(confDir); os.IsNotExist(err) {
		os.MkdirAll(confDir, 0o700)
	}

	// make sure the file exists
	if _, err := os.Stat(confPath); os.IsNotExist(err) {
		os.OpenFile(confPath, os.O_CREATE, 0o600)
	}

	return confPath
}
